var Express = require('express');
var BodyParser = require('body-parser');
var HttpStatus = require('http-status-codes');
var kStatus = require('../engine/kStatus.js');

var constructRouter = function(engine) {
    var router = Express.Router();
    var jsonParser = BodyParser.json();

    router.get('/', function(req, res) {
        var stats = engine.sessions.getStats();

        res.json({
            capacity: stats.capacity,
            activeCount: stats.activeCount,
            waitingCount: stats.waitingCount,
            availableEta: -1
        });
    });

    router.post('/', jsonParser, function(req, res) {
        try {
            if (!('scenarioId' in req.body)) {
                throw {status: kStatus.ERROR_PARAMETER};
            }

            var session = engine.sessions.createSession(req.body.scenarioId, req.ip);
            var sessionLog = session.log;

            if (sessionLog) {
                sessionLog.setUser(req.ip, req.headers['user-agent']);
            }

            var sessionUrl = req.protocol + '://' + req.get('host') +
                req.originalUrl + '/' + session.id;
            res.json({
                status: kStatus.OK,
                sessionId: session.id,
                sessionUrl: sessionUrl
            });
        } catch (err) {
            res.status(HttpStatus.INTERNAL_SERVER_ERROR).json({
                status: err.status
            });
        }
    });

    router.get('/:id', function(req, res) {
        var session = getSession(req, res);
        if (!session) return;

        session.refreshConnection();

        var results = {
            scenarioId: session.scenarioId,
            state: session.state,
            error: session.error,
            queuePosition: session.queuePosition,
            expiryTimer: session.activityTimeRemaining
        };

        if (session.isStarted()) {
            results.ports =  {
                web: session.ports.web,
                control: session.ports.control,
                upgrade: session.ports.upgrade,
                health: session.ports.health,
                privateData: session.ports.privateData,
                publicData: session.ports.publicData
            }
        }

        results.status = kStatus.OK;

        res.json(results);
    });

    router.delete('/:id', function (req, res) {
        var session = getSession(req, res);
        if (!session) return;

        engine.sessions.destroySession(session);
        res.json({status: kStatus.OK});
    });

    router.get('/:id/config', function(req, res) {
        var session = getSession(req, res);
        if (!session) return;

        if (!session.isStarted()) {
            res.status(HttpStatus.INTERNAL_SERVER_ERROR).json({
                status: kStatus.ERROR_STATE
            });
        } else {
            session.emulator.getStatus(function(error, result) {
                if (error) {
                    res.status(HttpStatus.INTERNAL_SERVER_ERROR).json({
                        status: error.status
                    });
                } else {
                    res.json({
                        status: kStatus.OK,
                        replayProtectionEnabled: result.replayProtectionEnabled
                    });
                }
            });
        }
    });

    router.put('/:id/config', jsonParser, function(req, res) {
        var session = getSession(req, res);
        if (!session) return;

        // TODO: use a promise chain for multiple commands. For now there is
        // only one command.

        session.kickActivity();

        if ('replayProtectionEnabled' in req.body) {
            var enabled = req.body.replayProtectionEnabled;
            session.emulator.enableReplayProtection(enabled, function(error, result) {
                if (error) {
                    res.status(HttpStatus.INTERNAL_SERVER_ERROR).json({
                        status: error.status
                    });
                } else {
                    res.json({status: kStatus.OK});
                }
            });
        }
        else {
            res.json({status: kStatus.OK});
        }
    });

    router.post('/:id/command', jsonParser, function(req, res) {
        var session = getSession(req, res);
        if (!session) return;

        var commandFunction = null;

        if ('command' in req.body) {
            if (req.body.command == 'restart') {
                commandFunction = restartCommand;
            } else if (req.body.command == 'ping') {
                commandFunction = pingCommand;
            }
        }

        if (!commandFunction) {
            res.status(HttpStatus.INTERNAL_SERVER_ERROR).json({
                status: kStatus.ERROR_PARAMETER
            });
        } else {
            session.kickActivity();
            commandFunction(session, req, res);
        }
    });

    var restartCommand = function(session, req, res) {
        engine.sessions.restartSession(session, function(error, result) {
            if (error) {
                res.status(HttpStatus.INTERNAL_SERVER_ERROR).json({
                    status: error.status
                });
            } else {
                res.json({status: kStatus.OK});
            }
        });
    };

    var pingCommand = function(session, req, res) {
        session.refreshConnection();
        res.json({status: kStatus.OK});
    };

    var getSession = function(req, res) {
        var session = engine.sessions.findSession(req.params.id);

        if (session == null) {
            res.status(HttpStatus.NOT_FOUND).json({
                status: kStatus.ERROR_NOT_FOUND
            });
        } else {
            session.log.setIp(req.ip);
        }

        return session;
    };

    return router;
};

module.exports = constructRouter;
